SUBROUTINE root ( f, x1, x2, dx, aroot, error )
!
!  Purpose:
!    To find a root of function f(x) between x1 and 
!    x2, searching with step sizes of dx.
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    02/16/96    S. J. Chapman        Original code
!
IMPLICIT NONE

! Declare calling arguments
REAL, EXTERNAL :: f             ! Function to find roots of
REAL, INTENT(IN) :: x1          ! Starting point for search
REAL, INTENT(IN) :: x2          ! Ending point for search
REAL, INTENT(IN) :: dx          ! Step size
REAL, INTENT(OUT) :: aroot      ! Root of function.
INTEGER, INTENT(OUT) :: error   ! Error flag:
                                !   0 = No error.
                                !   1 = x1 > x2.
                                !   2 = No root found

! Declare parameter:
REAL, PARAMETER :: epsilon = 1.0E-5  ! Convergence criterion

! Declare local variables:
REAL :: den                  ! Denominator  of conv. function
LOGICAL :: l_val_a           ! True if f(xa) > 0
LOGICAL :: l_val_b           ! True if f(xb) > 0
LOGICAL :: l_val_m           ! True if f(xm) > 0
INTEGER :: i                 ! Index variable
INTEGER :: n                 ! Number of steps to search
REAL :: val_a, val_b, val_m  ! f(xa), f(xb), f(xm)
REAL :: xa, xb, xm           ! Start, middle, end of interval

! First, check to make sure that the interval to search is valid.
error_chk: IF ( x1 > x2 ) THEN
   error = 1           ! Error--Starting point after end.
   aroot = 0.
ELSE

   ! Initialize error flag to 2, since no root found yet
   error = 2
   aroot = 0.
 
   ! Get number of steps to search over for sign change.
   n = NINT( (x2-x1) / dx + 1. )
 
   ! Now search for a zero crossing.  Get starting value 
   ! at first step. 
   xa = x1
   val_a = f(xa)
   l_val_a = val_a > 0.

   ! Search for a sign change between x1 and x2.
   outer: DO i = 1, n
 
      ! Get value at end of interval.
      xb = MIN( x1 + REAL(i-1) * dx, x2 )
      val_b  = f(xb)
      l_val_b = val_b > 0.    
 
      ! Is there a sign change in this interval?
      sign_change: IF ( l_val_a .NEQV. l_val_b ) THEN 
 
         ! There was a sign change in the interval.  The
         ! root is somewhere between xa and xb.  Process
         ! it in a WHILE loop.
         inner: DO
  
            ! Get value at midpoint.  
            xm = (xa + xb) / 2.
            val_m  = f(xm)
            l_val_m = val_m > 0.

            ! Test for convergence.
            den = MAX (1., xm)     ! Avoid problems at xm = 0.
            IF (ABS((xb - xa) / den) <= epsilon) THEN
               aroot = xm
               error = 0
               EXIT outer

            ELSE IF (l_val_a .EQV. l_val_m) THEN
               ! The sign change was in the second half.
               xa = xm
               val_a = val_m
               l_val_a = l_val_m

            ELSE
               ! The sign change was in the first half.
               xb = xm
               val_b = val_m
               l_val_b = l_val_m
            END IF
         END DO inner
      END IF sign_change

      ! We are still searching for a sign change here.
      ! Set new starting point for next interval.
      xa = xb
      val_a = val_b
      l_val_a = l_val_b
   END DO outer
END IF error_chk

END SUBROUTINE root
